#!/bin/bash -e
set -o pipefail
ZSDIR=$(mktemp -d)
export ZSDIR

echo "Test 'zs -c'"
./zs -c

echo
echo "Test 'zs -e and -d (encode/decode)'"
E=$(mktemp)
D=$(mktemp)
./zs -e < example > $E
./zs -d <$E|tee $D

cmp example $D
echo --
wc -m example
echo --
cat $E
echo --
wc -m $E

echo
echo test -E
rm -v $ZSDIR/known_keys
S=$(mktemp)
./zs -E < example2|tee $S
chmod u+x $S
PATH=.:$PATH
R="$($S)"
[ "x$R" = "x42" ];

# test broken messages

B=$(mktemp)
readchars=$(mktemp)
cat > $readchars <<EOF
(define (main args)
  (let1 chars (string->number (cadr args))
    (dotimes (i chars)
      (write-char (read-char))))
  0)
EOF

function test-failure(){
    echo
    C=$1
    echo "Test failure after $C chars"
    cat $E | {
        gosh $readchars $C
        tr S T
    } > $B
    cat $B
    echo
    if ./zs -d < $B; then
        echo Should fail.
        exit 1
    else
        echo "Got expected failure with exit status $?."
    fi
}

# note: the problem is we don't know the exact number of bytes used
# for the encoded public key and the signature.  But we know that 32
# chars are used for the public key and 64 chars are used for the
# signature.
test-failure $((2+16))
test-failure $((2+32+32))
test-failure $((2+32+64))
test-failure $((2+32+64+70))

echo
echo "Make sure script isn't executed if verification fails"
SE=$(mktemp -d)
SG=$(mktemp)
./zs -E <<EOF |tee $S
#!/bin/sh
touch $SE/executed
EOF
{
    head -n3 $S
    echo garbage
    echo _
} > $SG
chmod u+x $SG
$SG || true
if test -f $SE/executed; then
    echo "Script was still executed."
    exit 1
fi

echo
echo "Test single argument mode"
./zs -e > $E <<EOF 
#!/bin/sh
echo hello
EOF
[ "x$(./zs $(cat $E)|tail -n1)" = "xhello" ]

echo
echo "Make sure script isn't executed if key is untrusted"
rm -v $ZSDIR/known_keys
./zs -E <<EOF|tee $S
#!/bin/sh
touch $SE/executed
EOF
chmod u+x $S
$S || true

if test -f $SE/executed; then
    echo "Script was still executed."
    exit 1
fi

echo
echo "Make sure old header still works"
if ! [ "x$(./zs zsŅäŷUŽŚÝöÄŝĚřĸĸłĦŧğĚNÔXũWŬĽOİŒŪVBÑõŢķJÙĔðëkNœĿGžūĊŞĉĝŃþţJńčĪuĪĚ0ÍoÍĠŝŰŶĶĒĎYÂŶŅČļĂĳųεŵąøŋŽņÃÇĔÓeβ1ÕÙŘÑŎŐŗlŒŤÌÏŒŌÚŌÊőōōū20)" = "xhello" ]; then
    echo "Old header doesn't work"
    exit 1
fi

# at least do some cleanup on success
rm -vf $E $D $S $B $readchars $SG
rm -vrf $ZSDIR $SE

echo All tests passed.
