;;;
;;; Test kyotocabinet
;;;

(use gauche.test)
(use gauche.dictionary)

(test-start "kyotocabinet")
(use dbm.kyotocabinet)
(test-module 'dbm.kyotocabinet)

(define *db-file* "test.kch")
(sys-unlink *db-file*)

(test* "class"
       <kyotocabinet>
       (class-of (make <kyotocabinet>)))

(test* "low-level API"
       '("42" "43")
       (let1 db (kcdbopen *db-file* (logior KCOWRITER KCOCREATE))
         (kcdbset db "key" "42")
         (let1 r1 (kcdbget db "key")
           (kcdbset db "key" "43")
           (let1 r2 (kcdbget db "key")
             (kcdbclose db)
             (list r1 r2)))))

(test* "low-level API cursor"
       '("key" "43")
       (let1 db (kcdbopen *db-file* (logior KCOWRITER KCOCREATE))
         (let1 c (kcdbcursor db)
           (kccurjump c)
           (kccurget c 0)
           (unwind-protect
            (kccurget c 1)
            (kcdbclose db)))))

(test* "high-level API (get value)"
       "43"
       (let* ((db (dbm-open <kyotocabinet> :path *db-file*))
              (r (dbm-get db "key")))
         (dbm-close db)
         r))

(test* "high-level API (get updated value)"
       "42"
       (let1 db (dbm-open <kyotocabinet> :path *db-file*)
         (dbm-put! db "key" "42")
         (unwind-protect (dbm-get db "key")
                         (dbm-close db))))

(test* "high-level API (test existance)"
       '(#t #f)
       (let1 db (dbm-open <kyotocabinet> :path *db-file*)
         (unwind-protect (list
                          (dbm-exists? db "key")
                          (dbm-exists? db "nokey"))
                         (begin
                           (dbm-delete! db "key")
                           (dbm-close db)))))

(test* "high-level API (delete non-existing key)"
       (test-error)
       (let1 db (dbm-open <kyotocabinet> :path *db-file*)
         (unwind-protect (dbm-delete! db "key")
                         (dbm-close db))))

(test* "high-level API (closed?)"
       #t
       (let1 db (dbm-open <kyotocabinet> :path *db-file*)
         (dbm-close db)
         (dbm-closed? db)))

(test* "high-level API (already closed)"
       (test-error)
       (let1 db (dbm-open <kyotocabinet> :path *db-file*)
         (dbm-close db)
         (dbm-closed? db)
         (dbm-put! db "key" "42")))

(test* "high-level API (#f value)"
       #f
       (let1 db (dbm-open <kyotocabinet> :path *db-file* :value-convert #t)
         (dbm-put! db "key" #f)
         (unwind-protect
          (dbm-get db "key")
          (dbm-close db))))

(test* "high-level API (#f value)"
       #t
       (let1 db (dbm-open <kyotocabinet> :path *db-file* :value-convert #t)
         (unwind-protect
          (dbm-exists? db "key")
          (begin
            (dbm-delete! db "key")
            (dbm-close db)))))

(let1 alist '((a . 1) (b . 2))
  (test* "high-level API (fold)"
         alist
         (let1 db (dbm-open <kyotocabinet> :path *db-file* :key-convert #t :value-convert #t)
           (for-each (lambda(kv)
                       (dbm-put! db (car kv) (cdr kv)))
                     alist)
           (unwind-protect
            (reverse (dbm-fold db acons '()))
            (dbm-close db)))))

(test* "null bytes"
       "value\0with\0"
       (let1 db (dbm-open <kyotocabinet> :path *db-file*)
         (dbm-put! db "key\0withnull\0" "value\0with\0")
         (unwind-protect (dbm-get db "key\0withnull\0")
                         (dbm-close db))))

(test* "dictionary api"
       1
       (let1 db (dbm-open <kyotocabinet> :path *db-file* :key-convert #t :value-convert #t)
         (unwind-protect
          (dict-get db 'a)
          (dbm-close db))))

;; If you don't want `gosh' to exit with nonzero status even if
;; test fails, pass #f to :exit-on-failure.
(test-end :exit-on-failure #t)
