;;;
;;; kyotocabinet[1] bindings for gauche scheme[2].
;;; 
;;; [1] http://fallabs.com/kyotocabinet/
;;; [2] http://practical-scheme.net/gauche/
;;;
;;;   Copyright (c) 2015 Jens Thiele <karme@karme.de>
;;;   
;;;   Redistribution and use in source and binary forms, with or without
;;;   modification, are permitted provided that the following conditions
;;;   are met:
;;;   
;;;   1. Redistributions of source code must retain the above copyright
;;;      notice, this list of conditions and the following disclaimer.
;;;  
;;;   2. Redistributions in binary form must reproduce the above copyright
;;;      notice, this list of conditions and the following disclaimer in the
;;;      documentation and/or other materials provided with the distribution.
;;;  
;;;   3. Neither the name of the authors nor the names of its contributors
;;;      may be used to endorse or promote products derived from this
;;;      software without specific prior written permission.
;;;  
;;;   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
;;;   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
;;;   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
;;;   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
;;;   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
;;;   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
;;;   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
;;;   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
;;;   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;;   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;;   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
;;;  

(define-module dbm.kyotocabinet
  (extend dbm)
  (export <kyotocabinet>
          kcdbopen
          kcdbset
          kcdbget
          kcdbcheck
          kcdbcursor
          kccurjump
          kccurget
          kcdbclose
          kcdbclosed?
          KCOREADER
          KCOWRITER
          KCOCREATE
          KCONOLOCK
          KCOAUTOSYNC
          ))
(select-module dbm.kyotocabinet)

;;;
;;; High-level dbm interface
;;;

(define-class <kyotocabinet-meta> (<dbm-meta>)
  ())

(define-class <kyotocabinet> (<dbm>)
  ((sync      :init-keyword :sync   :initform #f)
   (nolock    :init-keyword :nolock :initform #f)
   (db :initform #f))
  :metaclass <kyotocabinet-meta>)

(define-method dbm-open ((self <kyotocabinet>))
  (next-method)
  (unless (slot-bound? self 'path)
    (error "path must be set to open kyotocabinet database"))
  (let* ([path   (slot-ref self 'path)]
         [rwmode (slot-ref self 'rw-mode)]
         [sync   (slot-ref self 'sync)]
         [nolock (slot-ref self 'nolock)]
         [rwopt  (case rwmode
                   [(:read) KCOREADER]
                   [(:write) (+ KCOWRITER
                                (if sync KCOAUTOSYNC 0)
                                (if nolock KCONOLOCK 0))]
                   [(:create) (+ KCOWRITER KCOCREATE
                                 (if sync KCOAUTOSYNC 0)
                                 (if nolock KCONOLOCK 0))])]
         [db     (kcdbopen path rwopt)])
    (slot-set! self 'db db)
    self))

(define-method dbm-close ((self <kyotocabinet>))
  (kcdbclose (slot-ref self 'db)))

(define-method dbm-put! ((self <kyotocabinet>) key value)
  (next-method)
  (when (zero? (kcdbset (slot-ref self 'db)
                        (%dbm-k2s self key)
                        (%dbm-v2s self value)))
    (error "dbm-put! failed" self)))

(define-method dbm-get ((self <kyotocabinet>) key . args)
  (next-method)
  (cond [(kcdbget (slot-ref self 'db) (%dbm-k2s self key))
         => (cut %dbm-s2v self <>)]
        [(pair? args) (car args)]     ;fall-back value
        [else  (errorf "kyotocabinet: no data for key ~s in database ~s"
                       key (slot-ref self 'db))]))

(define-method dbm-exists? ((self <kyotocabinet>) key . args)
  (not (= (kcdbcheck (slot-ref self 'db) (%dbm-k2s self key)) -1)))

(define-method dbm-delete! ((self <kyotocabinet>) key)
  (next-method)
  (when (zero? (kcdbdelete (slot-ref self 'db) (%dbm-k2s self key)))
    (errorf "dbm-delete!: deleteting key ~s from ~s failed" key self)))

(define-method dbm-closed? ((self <kyotocabinet>))
  (or (not (slot-ref self 'db))
      (kcdbclosed? (slot-ref self 'db))))

(define-method dbm-fold ((self <kyotocabinet>) proc knil)
  (let1 c (kcdbcursor (slot-ref self 'db))
    (kccurjump c)
    (let loop ([kv (kccurget c 1)] [r knil])
      (if kv
        (loop (kccurget c 1)
              (proc (%dbm-s2k self (car kv)) (%dbm-s2v self (cadr kv)) r))
        r))))

(define-method dbm-db-exists? ((class <kyotocabinet-meta>) name)
  ;; todo: better test?!
  (file-exists? name))

(define-method dbm-db-remove ((class <kyotocabinet-meta>) name)
  ;; todo: test wether it is a valid db first?!
  ;; might also be a directory
  (sys-unlink name))

;;;
;;; Low-level bindings
;;;

(inline-stub
 (declcode
  (.include "kclangc.h")
  "typedef struct ScmKCDBRec {
     SCM_HEADER;
     KCDB* db;
   } ScmKCDB;"
  "typedef struct ScmKCCURRec {
     SCM_HEADER;
     KCCUR* cur;
   } ScmKCCUR;")

 (define-cclass <kyotocabinet-db> :private ScmKCDB* "Scm_KCDBClass" ()
   ()
   [printer
    (Scm_Printf port "#<kyotocabinet-db %p>" (-> (SCM_KYOTOCABINET_DB obj) db))])

 (define-cfn scm_kcdb_finalize (obj data::void*) ::void :static
   (let* ((db::(ScmKCDB *) (SCM_KYOTOCABINET_DB obj)))
     (when (-> db db)
       (kcdbclose (-> db db))
       (kcdbdel (-> db db))
       (set! (-> db db) NULL))))
 
 (define-cproc kcdbopen (name::<string> opt::<int>)
   (let* ([z::ScmKCDB* (SCM_NEW ScmKCDB)])
     (SCM_SET_CLASS z (& Scm_KCDBClass))
     (Scm_RegisterFinalizer (SCM_OBJ z) scm_kcdb_finalize NULL)
     (set! (-> z db) (kcdbnew))
     (when (== (kcdbopen (-> z db) (Scm_GetString name) opt) 0)
       (Scm_Error "couldn't open kyotocabinet db %S" name))
     (return (SCM_OBJ z))))

 (define-cproc kcdbset (db::<kyotocabinet-db> key::<string> val::<string>) ::<int>
   (let* ((tmpk :: (const ScmStringBody*) (SCM_STRING_BODY key))
          (tmpv :: (const ScmStringBody*) (SCM_STRING_BODY val)))
     (result (kcdbset (-> db db)
                      (cast char* (SCM_STRING_BODY_START tmpk))
                      (SCM_STRING_BODY_SIZE tmpk)
                      (cast char* (SCM_STRING_BODY_START tmpv))
                      (SCM_STRING_BODY_SIZE tmpv)))))
 
 (define-cproc kcdbget (db::<kyotocabinet-db> key::<string>)
   (let* ((tmpk::(const ScmStringBody*) (SCM_STRING_BODY key))
          (vsiz::size_t 0)
          (r::(void*) (kcdbget (-> db db)
                               (cast (const char*) (SCM_STRING_BODY_START tmpk))
                               (SCM_STRING_BODY_SIZE tmpk) (& vsiz)))
          (scmret SCM_FALSE))
     (cond [(== NULL r)
            (return SCM_FALSE)]
           [else
            (set! scmret (Scm_MakeString r vsiz -1 SCM_STRING_COPYING))
            (kcfree r)
            (return scmret)])))
 
 (define-cproc kcdbcheck (db::<kyotocabinet-db> key::<string>) ::<int>
   (let* ((tmpk::(const ScmStringBody*) (SCM_STRING_BODY key)))
     (result (kcdbcheck (-> db db)
                        (cast (const char*) (SCM_STRING_BODY_START tmpk))
                        (SCM_STRING_BODY_SIZE tmpk)))))

 (define-cproc kcdbdelete (db::<kyotocabinet-db> key::<string>) ::<int>
   (let* ((tmpk::(const ScmStringBody*) (SCM_STRING_BODY key)))
     (result (kcdbremove (-> db db)
                         (cast (const char*) (SCM_STRING_BODY_START tmpk))
                         (SCM_STRING_BODY_SIZE tmpk)))))

 ;; todo: nearly duplicate of scm_kcdb_finalize
 (define-cproc kcdbclose (db::<kyotocabinet-db>)
   ;; (Scm_Printf SCM_CURERR "closing #<kyotocabinet-db %p>" (-> db db))
   (let* ((ret (SCM_MAKE_INT (kcdbclose (-> db db)))))
     (kcdbdel (-> db db))
     (set! (-> db db) NULL)
     ;; (Scm_Printf SCM_CURERR
     ;;             "ret=%S\n"
     ;;             ret)
     (return ret)))

 (define-cproc kcdbclosed? (db::<kyotocabinet-db>) ::<boolean>
   (result (== (-> db db) NULL)))

 (define-cclass <kyotocabinet-cursor> :private ScmKCCUR* "Scm_KCCURClass" ()
   ()
   [printer
    (Scm_Printf port "#<kyotocabinet-cursor %p>" (-> (SCM_KYOTOCABINET_CURSOR obj) cur))])

 (define-cproc kcdbcursor (db::<kyotocabinet-db>)
   (let* ((z::ScmKCCUR* (SCM_NEW ScmKCCUR)))
     (SCM_SET_CLASS z (& Scm_KCCURClass))
     ;; todo: leak / kccurdel
     ;; (Scm_RegisterFinalizer (SCM_OBJ z) gdbm_finalize NULL)
     (set! (-> z cur) (kcdbcursor (-> db db)))
     (return (SCM_OBJ z))))

 (define-cproc kccurjump (cur::<kyotocabinet-cursor>)
   (kccurjump (-> cur cur)))

 (define-cproc kccurget (cur::<kyotocabinet-cursor> step::<int>)
   (let* ((ksp::size_t 0)
          (vsp::size_t 0)
          (vbp::(const char*) NULL)
          (kbuf::(char*) (kccurget (-> cur cur)
                                   (& ksp)
                                   (& vbp)
                                   (& vsp)
                                   step))
          (scmret SCM_FALSE))
     (cond [(== NULL kbuf)
            (return SCM_FALSE)]
           [else
            (set! scmret (SCM_LIST2
                          (Scm_MakeString kbuf ksp -1 SCM_STRING_COPYING)
                          (Scm_MakeString vbp vsp -1 SCM_STRING_COPYING)))
            (kcfree kbuf)
            (return scmret)])))
 
 (define-enum KCOREADER)
 (define-enum KCOWRITER)
 (define-enum KCOCREATE)
 (define-enum KCOAUTOSYNC)
 (define-enum KCONOLOCK)
 )
